import Solitaire from './Solitaire.js';

export default class App{

    constructor(Game){
        window.APP = this;              // Raccourci pour accéder à notre instance d'application depuis n'importe où
        this.game = new Game();         // Création d'une instance du jeu que l'on souhaite afficher
        
        this.canvas = document.querySelector("#app");
        this.canvas.width = this.game.canvasSize;
        this.canvas.height = this.game.canvasSize;
        this.context = this.canvas.getContext("2d");
        this.context.imageSmoothingEnabled = false;
        this.images = {};

        this.sounds = {};
        this.canvas.onclick = this.handleClick;
        window.onresize = this.resizeCanvas;
        this.resizeCanvas();
        requestAnimationFrame(this.gameLoop);
        this.loadImages(this.game.images);   // On donne en paramètre la liste des images à télécharger
    }

    // Méthodes liées aux images
    loadImages(listImages) {
        this.listImages = listImages;   // On sauvegarde temporairement la liste des images à télécharger
        for(let imgName of listImages){ 
            const img = new Image();
            img.onload = () => this.saveImage(imgName, img); 
            img.src = './images/'+ imgName; 
        }
    }
    saveImage(nameComplet, img) { 
        const name = nameComplet.split(".");  
        this.images[name[0]] = img; 
        if(this.allImagesLoaded()){ 
            try{ this.game.handleLoaded(); }
            catch(error){}
        }  
    }
    allImagesLoaded() {                                                   
        for(let nameComplet of this.listImages){            // On passe en revue chaque images que l'on doit télécharger 
            const name = nameComplet.split(".")[0];         // On prend la partie du nom qui se situe avant l'extension
            if(!this.images[name]){ return false; }         // Si l'une des images n'est pas encore stockée, on renvoie false.
        }
        delete this.listImages;                             // Si toutes les images ont été trouvées, on supprime la liste des images à télécharger
        return true;                                        // Et on renvoie true
    }     

    // Méthodes liés aux sons
    loadSound(listSound) {
        this.listSound = listSound; 
        for(let soundName of listSound){ 
            const sound = new Audio('./sound/'+soundName);
            sound.oncanplaythrough = () => this.saveSound(soundName, sound); 
        }
    }
    saveSound(nameComplet, sound) { 
        sound.oncanplaythrough = null;
        const name = nameComplet.split(".");  
        this.sounds[name[0]] = sound; 
        if(this.allSoundLoaded()){ 
            try{ this.game.handleLoaded(); }
            catch(error){}
        }  
    }
    allSoundLoaded() {                                          
        for(let nameComplet of this.listSound){            
            const name = nameComplet.split(".")[0];       
            if(!this.sounds[name]){ return false; }       
        }
        delete this.listSound;                             
        return true;                                     
    }     


    // Methodes liées aux événements
    resizeCanvas = () => {
        const body = document.querySelector('body');
        const scale = Math.min((body.clientHeight / this.canvas.height), (body.clientWidth / this.canvas.width));
        this.canvas.style.transform = 'scale('+scale+')';
    }

    handleClick = (event) => {
        if(event.target !== this.canvas) return;
        event.preventDefault(); 
        let x = event.offsetX;
        let y = event.offsetY;          
        // On essaie d'appeler la fonction clicEvent de notre jeu en lui fournissant les coordonnées x et y
        try{ this.game.handleClick(x, y); } 
        catch(error){ /* Action à réaliser si notre jeu ne possède pas de fonction handleClick */ }
    }

    // Methods d'animation    
    gameLoop = (timestamp) => {
        requestAnimationFrame(this.gameLoop);
        const delta = this.getDelta(timestamp);
        if(!delta) return;
        this.isUpdated = false;
        try{ this.game.update(delta); }catch(error){}
        if(!this.isUpdated) return;
        try{ this.game.render(this.context); }catch(error){}
    }
    getDelta(timestamp) {
        const fpsMax = this.game.fps || 30;
        if(!this.prevTimestamp){ this.prevTimestamp = timestamp; }
        const elapsed = (timestamp - this.prevTimestamp);
        if(elapsed > (1000 / fpsMax)){
            this.prevTimestamp = timestamp;
            return Math.min((elapsed / 1000), 0.1);
        }
        return 0;
    }
    //Courbe d'accélération linéaire pour animation (de 0 a 1 par defaut)
    easeLinear = (time, delay, start=0, end=1) => {
        if(start > end){ return end; }
        return end * time / delay + start;
    }
    //Courbe d'accélération élastique pour animation (de 0 a 1 par defaut)
    easeOutElastic = (time, delay, start=0, end=1) => {
        if(start > end){ return end; }
        let s = 1.70158;
        let p = 0;
        let a = end;
        if (time == 0) return start;
        if ((time /= delay) == 1) return start + end;
        if (!p) p = delay * .3;
        if (a < Math.abs(end)) {
            a = end;
            s = p / 4;
        }
        else s = p / (2 * Math.PI) * Math.asin(end / a);
        return a * Math.pow(2, -10 * time) * Math.sin((time * delay - s) * (2 * Math.PI) / p) + end + start;
    }
    //Courbe rapide / lent / rapide
    easeInOutQuad = (time, delay, start=0, end=1) => {
        if(start > end){ return end; }
        if((time /= delay / 2) < 1) return end / 2 * time * time + start;
        return - end / 2 * ((--time) * (time - 2) - 1) + start;
    }

}

window.onload = new App(Solitaire);